'use client';
import { useInventory } from '../providers/inventory-provider';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  calculateClosingStockValue,
  calculatePotentialProfit,
  getReorderAlerts,
  getTopProductsBySales,
  calculateProductMetrics,
  getTopProductsByRevenue,
  calculateStockMovement,
  getFastVsSlowMovers,
} from '@/lib/inventory-calculations';
import { Bar, BarChart, XAxis, YAxis } from 'recharts';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import {
  FileX2,
  TrendingDown,
  TrendingUp,
  Zap,
  ArrowUpDown,
  AlertTriangle,
} from 'lucide-react';
import { ChartContainer, ChartTooltip, ChartTooltipContent } from '../ui/chart';
import { ScrollArea } from '../ui/scroll-area';
import { useState, useMemo } from 'react';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { CustomChartsView } from './custom-charts-view';

type ProductMetricsKeys =
  | 'Item Name'
  | 'SKU'
  | 'Product-wise Profit'
  | 'Profit Margin (%)'
  | 'Stock Movement';

type MoverKeys = 'Item Name' | 'SKU' | 'TurnoverRatio';

function KpiDashboard() {
  const { inventoryData } = useInventory();
  const [productSortConfig, setProductSortConfig] = useState<{
    key: ProductMetricsKeys;
    direction: 'ascending' | 'descending';
  } | null>({ key: 'Product-wise Profit', direction: 'descending' });

  const [moverSortConfig, setMoverSortConfig] = useState<{
    key: MoverKeys;
    direction: 'ascending' | 'descending';
  } | null>({ key: 'TurnoverRatio', direction: 'descending' });


  const formatInCrores = (value: number) => {
    if (value === 0) return '₹0';
    const crores = value / 10000000;
    if (Math.abs(crores) >= 1) {
      return `₹${crores.toFixed(2)} Crores`;
    }
    return `₹${value.toLocaleString('en-IN')}`;
  };

  const formatPercentage = (value: number) => `${value.toFixed(2)}%`;

  const closingValueCost = calculateClosingStockValue(inventoryData, 'cost');
  const closingValueSelling = calculateClosingStockValue(
    inventoryData,
    'selling'
  );
  const potentialProfit = calculatePotentialProfit(inventoryData);
  const reorderAlerts = getReorderAlerts(inventoryData);
  const topProductsBySales = getTopProductsBySales(inventoryData, 5);
  const topProductsByRevenue = getTopProductsByRevenue(inventoryData, 5);
  const productMetrics = calculateProductMetrics(inventoryData);
  const stockMovement = calculateStockMovement(inventoryData);
  const { fastMovers, slowMovers } = getFastVsSlowMovers(inventoryData);

  const sortedProductMetrics = useMemo(() => {
    let sortableItems = [...productMetrics].map((item) => ({
      ...item,
      'Stock Movement': stockMovement[item.SKU as string] || 0,
    }));
    if (productSortConfig !== null) {
      sortableItems.sort((a, b) => {
        const aValue = a[productSortConfig.key];
        const bValue = b[productSortConfig.key];

        if (typeof aValue === 'number' && typeof bValue === 'number') {
          return productSortConfig.direction === 'ascending'
            ? aValue - bValue
            : bValue - aValue;
        }

        if (typeof aValue === 'string' && typeof bValue === 'string') {
          return productSortConfig.direction === 'ascending'
            ? aValue.localeCompare(bValue)
            : bValue.localeCompare(aValue);
        }

        return 0;
      });
    }
    return sortableItems;
  }, [productMetrics, stockMovement, productSortConfig]);

  const sortedMovers = (
    data: typeof fastMovers,
    config: typeof moverSortConfig
  ) => {
    const sortableItems = [...data];
    if (config !== null) {
      sortableItems.sort((a, b) => {
        const aValue = a[config.key];
        const bValue = b[config.key];
        if (typeof aValue === 'number' && typeof bValue === 'number') {
          return config.direction === 'ascending'
            ? aValue - bValue
            : bValue - aValue;
        }
         if (typeof aValue === 'string' && typeof bValue === 'string') {
          return config.direction === 'ascending'
            ? aValue.localeCompare(bValue)
            : bValue.localeCompare(aValue);
        }
        return 0;
      });
    }
    return sortableItems;
  };

  const requestProductSort = (key: ProductMetricsKeys) => {
    let direction: 'ascending' | 'descending' = 'ascending';
    if (
      productSortConfig &&
      productSortConfig.key === key &&
      productSortConfig.direction === 'ascending'
    ) {
      direction = 'descending';
    }
    setProductSortConfig({ key, direction });
  };
  
  const requestMoverSort = (key: MoverKeys) => {
    let direction: 'ascending' | 'descending' = 'ascending';
    if (
      moverSortConfig &&
      moverSortConfig.key === key &&
      moverSortConfig.direction === 'ascending'
    ) {
      direction = 'descending';
    }
    setMoverSortConfig({ key, direction });
  };


  const getSortIndicator = (
    key: ProductMetricsKeys | MoverKeys,
    config: typeof productSortConfig | typeof moverSortConfig
  ) => {
    if (!config || config.key !== key) {
      return <ArrowUpDown className="ml-2 h-4 w-4 opacity-50" />;
    }
    return config.direction === 'ascending' ? ' ▲' : ' ▼';
  };

  return (
    <div className="grid auto-rows-max items-start gap-4 md:gap-8 lg:col-span-2">
      <div className="grid gap-4 sm:grid-cols-1 md:grid-cols-3 lg:grid-cols-3">
        <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 flex flex-col">
          <CardHeader>
            <CardTitle>Closing Stock (Cost)</CardTitle>
            <CardDescription>Total value at cost price.</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold font-heading">
              {formatInCrores(closingValueCost)}
            </div>
          </CardContent>
        </Card>
        <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 delay-100 flex flex-col">
          <CardHeader>
            <CardTitle>Closing Stock (Selling)</CardTitle>
            <CardDescription>Total value at selling price.</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold font-heading">
              {formatInCrores(closingValueSelling)}
            </div>
          </CardContent>
        </Card>
        <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 delay-200 flex flex-col">
          <CardHeader>
            <CardTitle>Potential Profit</CardTitle>
            <CardDescription>From closing stock.</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold font-heading">
              {formatInCrores(potentialProfit)}
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid gap-4 md:grid-cols-2 md:gap-8 lg:grid-cols-2">
        <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 delay-300 flex flex-col">
          <CardHeader>
            <CardTitle>Top 5 Products by Sales Units</CardTitle>
            <CardDescription>
              Your best-selling products by units sold.
            </CardDescription>
          </CardHeader>
          <CardContent className="flex-1">
            <ChartContainer
              config={{
                sales: {
                  label: 'Sales (Units)',
                  color: 'hsl(var(--chart-1))',
                },
              }}
              className="min-h-[300px] w-full"
            >
              <BarChart
                accessibilityLayer
                data={topProductsBySales}
                margin={{ left: 12, right: 12 }}
              >
                <XAxis
                  dataKey="Item Name"
                  tickLine={false}
                  axisLine={false}
                  tickMargin={8}
                  tickFormatter={(value) => value.slice(0, 15)}
                />
                <YAxis
                  stroke="#888888"
                  fontSize={12}
                  tickLine={false}
                  axisLine={false}
                  tickFormatter={(value) => `${value}`}
                />
                <ChartTooltip
                  cursor={false}
                  content={<ChartTooltipContent indicator="dot" />}
                />
                <Bar
                  dataKey="Sales (Units)"
                  fill="var(--color-sales)"
                  radius={4}
                />
              </BarChart>
            </ChartContainer>
          </CardContent>
        </Card>
        <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 delay-400 flex flex-col">
          <CardHeader>
            <CardTitle>Top 5 Products by Revenue</CardTitle>
            <CardDescription>
              Your best-selling products by total revenue (in ₹ Crores).
            </CardDescription>
          </CardHeader>
          <CardContent className="flex-1">
            <ChartContainer
              config={{
                revenue: {
                  label: 'Revenue',
                  color: 'hsl(var(--chart-2))',
                },
              }}
              className="min-h-[300px] w-full"
            >
              <BarChart
                accessibilityLayer
                data={topProductsByRevenue}
                margin={{ left: 12, right: 12 }}
              >
                <XAxis
                  dataKey="Item Name"
                  tickLine={false}
                  axisLine={false}
                  tickMargin={8}
                  tickFormatter={(value) => value.slice(0, 15)}
                />
                <YAxis
                  stroke="#888888"
                  fontSize={12}
                  tickLine={false}
                  axisLine={false}
                  tickFormatter={(value) => formatInCrores(value)}
                />
                <ChartTooltip
                  cursor={false}
                  content={<ChartTooltipContent indicator="dot" />}
                />
                <Bar
                  dataKey="Revenue"
                  fill="var(--color-revenue)"
                  radius={4}
                />
              </BarChart>
            </ChartContainer>
          </CardContent>
        </Card>
      </div>

      <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 delay-500">
        <CardHeader>
          <CardTitle>Inventory Health & Action Center</CardTitle>
          <CardDescription>
            An overview of your inventory's performance and what to do next.
          </CardDescription>
        </CardHeader>
        <CardContent className="grid gap-8 md:grid-cols-2">
          <div className="flex flex-col">
            <h4 className="font-semibold mb-2 flex items-center">
              <AlertTriangle className="w-4 h-4 mr-2 text-yellow-500" />
              Reorder Alerts
            </h4>
             <p className="text-sm text-muted-foreground mb-4">
              These items have fallen below their specified reorder level.
            </p>
            <ScrollArea className="h-48">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Product</TableHead>
                    <TableHead>SKU</TableHead>
                    <TableHead className="text-right">Stock</TableHead>
                    <TableHead className="text-right">Reorder Lvl</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {reorderAlerts.length > 0 ? (
                    reorderAlerts.map((item, index) => (
                    <TableRow key={`reorder-${item.SKU}-${index}`}>
                      <TableCell>{item['Item Name']}</TableCell>
                      <TableCell><Badge variant="outline">{item['SKU']}</Badge></TableCell>
                      <TableCell className="text-right font-medium text-destructive">{item['Closing Stock (Units)']}</TableCell>
                      <TableCell className="text-right">{item['Reorder Level']}</TableCell>
                    </TableRow>
                  ))
                  ) : (
                     <TableRow>
                        <TableCell colSpan={4} className="h-24 text-center">
                          No reorder alerts.
                        </TableCell>
                      </TableRow>
                  )}
                </TableBody>
              </Table>
            </ScrollArea>
          </div>

          <div className="grid gap-4 md:grid-cols-2">
            <div className="flex flex-col">
              <h4 className="font-semibold mb-2 flex items-center">
                <TrendingUp className="w-4 h-4 mr-2 text-green-500" />
                Fast Movers
              </h4>
              <ScrollArea className="h-48">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="cursor-pointer hover:bg-muted" onClick={() => requestMoverSort('Item Name')}>
                         <div className="flex items-center">Product {getSortIndicator('Item Name', moverSortConfig)}</div>
                      </TableHead>
                      <TableHead className="text-right cursor-pointer hover:bg-muted" onClick={() => requestMoverSort('TurnoverRatio')}>
                         <div className="flex items-center justify-end">Turnover{getSortIndicator('TurnoverRatio', moverSortConfig)}</div>
                      </TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {sortedMovers(fastMovers, moverSortConfig).slice(0, 10).map((item, index) => (
                      <TableRow key={`fast-${item.SKU}-${index}`}>
                        <TableCell>{item['Item Name']}</TableCell>
                        <TableCell className="text-right font-medium text-primary">
                          {item.TurnoverRatio.toFixed(2)}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </ScrollArea>
            </div>
            <div className="flex flex-col">
              <h4 className="font-semibold mb-2 flex items-center">
                <TrendingDown className="w-4 h-4 mr-2 text-red-500" />
                Slow Movers
              </h4>
              <ScrollArea className="h-48">
                <Table>
                   <TableHeader>
                    <TableRow>
                      <TableHead className="cursor-pointer hover:bg-muted" onClick={() => requestMoverSort('Item Name')}>
                         <div className="flex items-center">Product {getSortIndicator('Item Name', moverSortConfig)}</div>
                      </TableHead>
                      <TableHead className="text-right cursor-pointer hover:bg-muted" onClick={() => requestMoverSort('TurnoverRatio')}>
                         <div className="flex items-center justify-end">Turnover{getSortIndicator('TurnoverRatio', moverSortConfig)}</div>
                      </TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {sortedMovers(slowMovers, moverSortConfig).slice(0, 10).map((item, index) => (
                      <TableRow key={`slow-${item.SKU}-${index}`}>
                        <TableCell>{item['Item Name']}</TableCell>
                        <TableCell className="text-right font-medium text-muted-foreground">
                          {item.TurnoverRatio.toFixed(2)}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </ScrollArea>
            </div>
          </div>
        </CardContent>
      </Card>


      <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500 delay-600">
        <CardHeader>
          <CardTitle>Product Profitability &amp; Movement</CardTitle>
          <CardDescription>
            Detailed profit margins and stock movement for each product.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <ScrollArea className="h-96">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead
                    className="cursor-pointer hover:bg-muted"
                    onClick={() => requestProductSort('Item Name')}
                  >
                    <div className="flex items-center">
                      Product
                      {getSortIndicator('Item Name', productSortConfig)}
                    </div>
                  </TableHead>
                  <TableHead
                    className="cursor-pointer hover:bg-muted"
                    onClick={() => requestProductSort('SKU')}
                  >
                    <div className="flex items-center">
                      SKU
                      {getSortIndicator('SKU', productSortConfig)}
                    </div>
                  </TableHead>
                  <TableHead
                    className="text-right cursor-pointer hover:bg-muted"
                    onClick={() => requestProductSort('Product-wise Profit')}
                  >
                    <div className="flex items-center justify-end">
                      Profit
                      {getSortIndicator('Product-wise Profit', productSortConfig)}
                    </div>
                  </TableHead>
                  <TableHead
                    className="text-right cursor-pointer hover:bg-muted"
                    onClick={() => requestProductSort('Profit Margin (%)')}
                  >
                    <div className="flex items-center justify-end">
                      Margin
                      {getSortIndicator('Profit Margin (%)', productSortConfig)}
                    </div>
                  </TableHead>
                  <TableHead
                    className="text-right cursor-pointer hover:bg-muted"
                    onClick={() => requestProductSort('Stock Movement')}
                  >
                    <div className="flex items-center justify-end">
                      Stock Movement
                      {getSortIndicator('Stock Movement', productSortConfig)}
                    </div>
                  </TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {sortedProductMetrics.map((item, index) => (
                  <TableRow key={`${item.SKU}-profit-${index}`}>
                    <TableCell className="font-medium">
                      {item['Item Name']}
                    </TableCell>
                    <TableCell>
                      <Badge variant="outline">{item.SKU}</Badge>
                    </TableCell>
                    <TableCell className="text-right">
                      {formatInCrores(item['Product-wise Profit'])}
                    </TableCell>
                    <TableCell className="text-right">
                      <span
                        className={
                          item['Profit Margin (%)'] > 0
                            ? 'text-green-600'
                            : 'text-red-600'
                        }
                      >
                        {formatPercentage(item['Profit Margin (%)'])}
                      </span>
                    </TableCell>
                    <TableCell className="text-right">
                       <span className={ item['Stock Movement'] > 50 ? 'text-green-600' : item['Stock Movement'] < 10 ? 'text-red-600' : ''}>
                        {formatPercentage(item['Stock Movement'])}
                      </span>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </ScrollArea>
        </CardContent>
      </Card>
    </div>
  );
}

export function KpiVisualizations() {
  const { inventoryData, clearInventory, fileName } = useInventory();

  if (inventoryData.length === 0) {
    return null;
  }

  return (
    <>
      <div className="flex justify-between items-center">
        <div>
          <h2 className="font-heading text-2xl font-bold">Inventory Analysis</h2>
          <p className="text-muted-foreground">
            Showing insights for:{' '}
            <span className="font-semibold text-primary">{fileName}</span>
          </p>
        </div>
        <Button variant="outline" onClick={clearInventory}>
          <FileX2 className="mr-2 h-4 w-4" />
          Upload New File
        </Button>
      </div>

      <Tabs defaultValue="kpi-dashboard" className="mt-4">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="kpi-dashboard">KPI Dashboard</TabsTrigger>
          <TabsTrigger value="custom-charts">Custom Charts</TabsTrigger>
        </TabsList>
        <TabsContent value="kpi-dashboard" className="mt-4">
          <KpiDashboard />
        </TabsContent>
        <TabsContent value="custom-charts" className="mt-4">
          <CustomChartsView />
        </TabsContent>
      </Tabs>
    </>
  );
}
